<?php

namespace App\Sys\Repositories\Exams;

use App\Models\Exam;
use App\Models\Result;
use App\Models\Package;
use App\Models\Setting;
use App\Models\Subscription;

class ExamRepository
{
    protected $exam;

    public function __construct(Exam $exam)
    {
        $this->exam = $exam;
    }

    public function getLatestSubscriptionDate($user)
    {
        return Subscription::query()
            ->where('user_id', $user->id)
            ->where('expiration_date', '>', now())
            ->where('country_' . $user->countryCode, true)
            ->latest('subscription_date')
            ->value('subscription_date');
    }

    public function getActivePackageIds($user)
    {
        return Package::query()
            ->whereHas('subscriptions', function ($query) use ($user) {
                $query->where('user_id', $user->id)
                    ->where('country_' . $user->countryCode, true)
                    ->where('expiration_date', '>', now());
            })
            ->pluck('id');
    }

    public function getExamsBySubscribedPackages($limit, $activePackageId, $user)
    {
        return $this->exam->whereHas('packages', function ($query) use ($activePackageId) {
            $query->whereIn('package_id', $activePackageId);
        })
            ->where('country_' . $user->countryCode, true)
            ->paginate($limit);
    }

    public function getExamsBySubscribedPackagesWithResults($user, $activePackageId, $latestSubscriptionDate)
    {
        return Exam::whereHas('packages', function ($query) use ($activePackageId, $user) {
            $query->where('package_id', $activePackageId)
                ->where('country_' . $user->countryCode, true);
        })
            ->with(['results' => function ($query) use ($user, $latestSubscriptionDate) {
                $query->where('user_id', $user->id)
                    ->where('created_at', '>=', $latestSubscriptionDate)
                    ->where('country_' . $user->countryCode, true);
            }])
            ->where('country_' . $user->countryCode, true)
            ->get();
    }


    public function getCurrentAttempts($user, $latestSubscriptionDate)
    {
        return Result::query()
            ->where('user_id', $user->id)
            ->where('created_at', '>=', $latestSubscriptionDate)
            ->where('country_' . $user->countryCode, true)
            ->select('exam_id')
            ->selectRaw('COUNT(*) as attempt_count')
            ->groupBy('exam_id')
            ->pluck('attempt_count', 'exam_id');
    }

    public function getAttemptsUsed($user, $examId, $latestSubscriptionDate)
    {
        return Result::query()
            ->where('user_id', $user->id)
            ->where('country_' . $user->countryCode, true)
            ->where('exam_id', $examId)
            ->where('created_at', '>=', $latestSubscriptionDate)
            ->count();
    }

    public function getExamWithCategoryAndQuestions($examId)
    {
        $country = config('app.country');

        return Exam::with([
            'examCategory' => function ($query) use ($country) {
                $query->where('country_' . $country, true);
            },
            'examCategory.questions' => function ($query) use ($country) {
                $query->where('country_' . $country, true)
                    ->orderBy('arrangment', 'asc');
            },
            'examCategory.questions.answers' => function ($query) use ($country) {
                $query->where('country_' . $country, true);
            }
        ])->find($examId);
    }


    public function isExamInSubscribedPackages($user, $examId, $activePackageIds)
    {
        return Exam::whereHas('packages', function ($query) use ($activePackageIds, $user) {
            $query->whereIn('package_id', $activePackageIds);
        })
            ->where('id', $examId)
            ->where('country_' . $user->countryCode, true)
            ->exists();
    }

    public function getTestExam()
    {
        return  Setting::select('test_exam_id')->find(1);
    }

    public function createResultRecord($user, $exam, int $attemptsUsed)
    {
        $result = new Result();
        $result->user_id = $user->id;
        $result->exam_id = $exam->id;
        $result->attempt_num = $attemptsUsed + 1; //Current attempt number - Just for ref Won't be used to count attempts.
        $result->score = 0.00;
        $result->passed_exam = 0;
        $result->json_score = null;
        $result->total_current_questions = $exam->questions_num;
        $result->total_right_questions = 0;
        $result->total_wrong_questions = 0;
        $result->total_skiped_questions = 0;
        $result->total_not_answered_questions = $exam->questions_num;
        $result->total_flaged_questions = 0;
        $propertyName = 'country_' . $user->countryCode;
        $result->{$propertyName} = 1;
        $result->save();

        return $result;
    }

    public function updateResultRecord($request)
    {
        $result = Result::findOrFail($request['result_record']);

        // Calculate the score (handle division by zero)
        $totalQuestions = $result->total_current_questions;
        $score = $totalQuestions > 0 ? ($request['total_right_questions'] * 100) / $totalQuestions : 0;

        // Update the result instance properties
        $result->score = $score;
        $result->passed_exam = $request['passed_exam'];
        $result->json_score = isset($request['json_score']) ? $request['json_score'] : null;
        $result->total_right_questions = $request['total_right_questions'];
        $result->total_wrong_questions = $request['total_wrong_questions'];
        $result->total_skiped_questions = $request['total_skiped_questions'];
        $result->total_not_answered_questions = $request['total_not_answered_questions'];

        $result->save();

        return $result;
    }
}
