<?php

namespace App\Http\Controllers\Admin;

use App\Models\PackageVideoOffers;
use Validator;
use App\Models\Country;
use App\Models\PackageVideo;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;

class PackageVideoOfferController extends Controller
{
    public function index($packageId)
    {
        if (!PackageVideo::where('id', $packageId)->exists()) {
            return;
        }
        $countryCode = session('admin_country_code');
        $offers = PackageVideoOffers::where('package_id', $packageId)
            ->when($countryCode, function ($query) use ($countryCode) {
                return $query->where('country_' . $countryCode, 1);
            })
            ->latest()
            ->get();

        return view('admin.package_video.Offer.index', compact('offers', 'packageId'));
    }

    public function create(Request $request)
    {
        $package = PackageVideo::findOrFail($request->id);
        $countryCodes = Country::pluck('code')->toArray();

        $activeOffers = PackageVideoOffers::where('package_id', $package->id)
            ->where(function ($query) {
                $query->whereNull('end_date')
                    ->orWhere('end_date', '>=', now());
            })
            ->get();

        $disabledCountries = collect($countryCodes)
            ->filter(function ($code) use ($activeOffers) {
                $column = "country_{$code}";
                foreach ($activeOffers as $offer) {
                    if ($offer->{$column} != 0) {
                        return false;
                    }
                }
                return true;
            })
            ->values()
            ->toArray();

        $countriesToBeSelected = Country::whereIn('code', $disabledCountries)->get();

        return view('admin.package_video.Offer.create', [
            'package' => $package,
            'countries' => $countriesToBeSelected,
        ]);
    }

    public function edit(PackageVideo $package, PackageVideoOffers $offer)
    {
        $countries = Country::all();
        return view('admin.package_video.Offer.edit', compact('package', 'countries', 'offer'));
    }

    public function store(Request $request)
    {
        $rules = [
            'title_ar' => 'required',
            'package_id' => 'required',
            'start_date' => 'required',
            'end_date' => 'required',
            'discount_amount' => 'required'
        ];
        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            session()->flash('notif', trans('messages.NOt Save'));
            return redirect()->back()->withErrors($validator->errors())->withInput($request->all());
        }

        $selectedCountries = collect(Country::pluck('code'))
            ->mapWithKeys(function ($code) use ($request) {
                return ["country_$code" => $request->has("country_$code")];
            })
            ->filter()
            ->keys()
            ->toArray();

        $conflictingCountries = PackageVideoOffers::where('package_id', $request->package_id)
            ->where(function ($query) {
                $query->whereNull('end_date')
                    ->orWhere('end_date', '>=', now());
            })
            ->get()
            ->flatMap(function ($offer) use ($selectedCountries) {
                return collect($selectedCountries)->filter(function ($country) use ($offer) {
                    return $offer->{$country} == true;
                });
            })
            ->unique()
            ->values();

        if ($conflictingCountries->isNotEmpty()) {
            return redirect()->back()
                ->withErrors([
                    'country_conflict' => trans('messages.country_conflict_error'),
                    'conflicting_countries' => $conflictingCountries
                ])
                ->withInput($request->all());
        }

        $validCountryCodes = Country::pluck('code')->toArray();

        $offer = new PackageVideoOffers();
        $offer->title_ar = $request->title_ar;
        $offer->title_en = $request->title_en;
        $offer->title_nl = $request->title_nl;
        $offer->package_id  = $request->package_id;
        $offer->start_date = $request->start_date;
        $offer->end_date = $request->end_date;
        $offer->discount_amount = $request->discount_amount;
        foreach ($validCountryCodes as $code) {
            $columnName = 'country_' . $code;
            $offer->$columnName = $request->has('country_' . $code) ? 1 : 0;
        }
        $offer->save();
        session()->flash('notif', trans('messages.add_successfully'));
        return redirect()->back();
    }

    public function update(Request $request, PackageVideoOffers $offer)
    {
        $rules = [
            'title_ar' => 'required',
            'package_id' => 'required',
            'start_date' => 'required',
            'end_date' => 'required',
            'discount_amount' => 'required'
        ];
        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            session()->flash('notif', trans('messages.NOt Save'));
            return redirect()->back()->withErrors($validator->errors())->withInput($request->all());
        }

        $selectedCountries = collect(Country::pluck('code'))
            ->mapWithKeys(function ($code) use ($request) {
                return ["country_$code" => $request->has("country_$code")];
            })
            ->filter()
            ->keys()
            ->toArray();

        $conflictingCountries = PackageVideoOffers::where('package_id', $offer->package_id)
            ->where('id', '!=', $offer->id)
            ->where(function ($query) {
                $query->whereNull('end_date')
                    ->orWhere('end_date', '>=', now());
            })
            ->get()
            ->flatMap(function ($otherOffer) use ($selectedCountries) {
                return collect($selectedCountries)->filter(function ($country) use ($otherOffer) {
                    return $otherOffer->{$country} == true;
                });
            })
            ->unique()
            ->values();

        if ($conflictingCountries->isNotEmpty()) {
            return redirect()->back()
                ->withErrors([
                    'country_conflict' => trans('messages.country_conflict_error'),
                    'conflicting_countries' => $conflictingCountries
                ])
                ->withInput($request->all());
        }

        $validCountryCodes = Country::pluck('code')->toArray();

        $offer->title_ar = $request->title_ar;
        $offer->title_en = $request->title_en;
        $offer->title_nl = $request->title_nl;
        $offer->package_id = $request->package_id;
        $offer->start_date = $request->start_date;
        $offer->end_date = $request->end_date;
        $offer->discount_amount = $request->discount_amount;
        foreach ($validCountryCodes as $code) {
            $columnName = 'country_' . $code;
            $offer->$columnName = $request->has('country_' . $code) ? 1 : 0;
        }
        $offer->save();
        session()->flash('notif', trans('messages.updated_successfully'));
        return redirect()->back();
    }

    public function destroy(PackageVideoOffers $offer)
    {
        $offer->delete();
        session()->flash('notif', trans('messages.offer deleted successfully'));

        return redirect()->back();
    }
}
