<?php

namespace App\Sys\Services\Results;

use App\Models\PackageExam;
use App\Sys\Services;
use App\Sys\Services\Exams\ExamService;
use App\Sys\Repositories\Results\ProgressRepository;

class ProgressService extends Services
{
    protected $progressRepository;
    protected $examService;

    public function __construct(ProgressRepository $progressRepository, ExamService $examService)
    {
        $this->progressRepository = $progressRepository;
        $this->examService = $examService;
    }

    public function getCurrentExamPackageProgress($user)
    {
        $activePackageId = $this->examService->getActivePackageIds($user);

        if (empty($activePackageId)) {
            $this->setError('No active package found');
            return false;
        }

        $latestSubscriptionDate = $this->examService->getLatestSubscriptionDate($user);

        if (!$latestSubscriptionDate) {
            $this->setError('No subscription found');
            return false;
        }

        $results = $this->progressRepository->getResultsForExams($user, $latestSubscriptionDate);

        $groupedResults = collect($results)->groupBy('exam_id');

        $examsWithResults = $this->examService->getExamsBySubscribedPackagesWithResults($user, $activePackageId, $latestSubscriptionDate);

        if ($examsWithResults->isEmpty()) {
            $this->setError('No exams found for the active package');
            return false;
        }

        // Build flattened results
        $flattenedResults = [];

        foreach ($examsWithResults as $exam) {
            if ($groupedResults->has($exam->id)) {
                foreach ($groupedResults->get($exam->id) as $result) {
                    $flattenedResults[] = [
                        'exam_id' => $exam->id,
                        'exam_name' => $exam->examName,
                        'result_id' => $result->id,
                        'passed_exam' => $result->passed_exam,
                        'score' => $result->score,
                    ];
                }
            } else {
                $flattenedResults[] = [
                    'exam_id' => $exam->id,
                    'exam_name' => $exam->examName,
                    'result_id' => null,
                    'passed_exam' => null,
                    'score' => null,
                ];
            }
        }

        // Calculate progress summary
        $collect = collect($results)->pluck('exam_id')->toArray();
        $failedExamCount = collect($results)->where('passed_exam', '=', 0)->count();
        $passedExamCount = collect($results)->where('passed_exam', '=', 1)->count();
        $notResolvedExamsCount = PackageExam::where('package_id', $activePackageId)
            ->whereNotIn('exam_id', $collect)
            ->count();

        // Handle division by zero in percentage calculation
        $totalExams = $passedExamCount + $failedExamCount + $notResolvedExamsCount;
        $passedPercentage = $totalExams > 0
            ? round(($passedExamCount / $totalExams) * 100, 2) . '%'
            : '0%';

        $progressSummary = [
            'passed_exam_count' => $passedExamCount,
            'failed_exam_count' => $failedExamCount,
            'not_resolved_exams_count' => $notResolvedExamsCount,
            'passed_percentage' => $passedPercentage,
        ];

        return [
            'progress_summary' => $progressSummary,
            'exams_progress' => $flattenedResults,
        ];
    }
}
