<?php

namespace App\Sys\Services\Notifications;

use App\Sys\Services;
use Kreait\Firebase\Factory;
use Kreait\Firebase\Messaging\CloudMessage;
use App\Sys\Repositories\Notifications\NotificationRepository;

class NotificationService extends Services
{
    private $messaging;
    private $notificationRepository;

    public function __construct(NotificationRepository $notificationRepository)
    {
        $this->notificationRepository = $notificationRepository;
        $this->messaging = $this->initializeFirebaseMessaging();
    }

    private function initializeFirebaseMessaging()
    {
        return (new Factory)
            ->withServiceAccount(config('services.firebase.adminsdk'))
            ->createMessaging();
    }

    public function sendNotificationToTopic($request)
    {
        $validatedData = $request->validate([
            'title' => ['required', 'string', 'max:255'],
            'body' => ['required', 'string', 'max:255'],
            'action' => ['required', 'in:general,course,package_exam,package_video'],
            'topic' => ['required', 'in:ia,ae,sa'],
        ]);

        $pushNotification = $this->sendPushToTopic(
            $validatedData['topic'],
            $validatedData['title'],
            $validatedData['body'],
            $validatedData['action']
        );

        if (!$pushNotification) {
            return false;
        }

        return $this->notificationRepository->storeNotification($validatedData);
    }

    public function sendNotificationToUser($request)
    {
        $validatedData = $request->validate([
            'title' => ['required', 'string', 'max:255'],
            'body' => ['required', 'string', 'max:255'],
            'action' => ['required', 'in:general,course,package_exam,package_video'],
            'user_device_token' => ['required', 'string'],
        ]);

        $pushNotification =  $this->sendPushToUser(
            $validatedData['user_device_token'],
            $validatedData['title'],
            $validatedData['body'],
            $validatedData['action']
        );

        if (!$pushNotification) {
            return false;
        }

        return $this->notificationRepository->storeNotification($validatedData);
    }

    private function createNotificationPayload($title, $body, $action)
    {
        return [
            'notification' => [
                'title' => $title,
                'body' => $body,
            ],
            'data' => [
                'action' => $action,
                'title' => $title,
                'body' => $body,
            ],
        ];
    }

    private function sendPushToTopic($topic, $title, $body, $action)
    {
        try {
            $payload = $this->createNotificationPayload($title, $body, $action);
            $message = CloudMessage::fromArray(array_merge($payload, ['topic' => $topic]));
            return $this->messaging->send($message);
        } catch (\Exception $e) {
            return false;
        }
    }

    private function sendPushToUser($deviceToken, $title, $body, $action)
    {
        try {
            $payload = $this->createNotificationPayload($title, $body, $action);
            $message = CloudMessage::fromArray(array_merge($payload, ['token' => $deviceToken]));
            return $this->messaging->send($message);
        } catch (\Exception $e) {
            return false;
        }
    }
}
