<?php

namespace App\Sys\Services\Exams;

use App\Sys\Services;
use Illuminate\Support\Facades\Validator;
use App\Sys\Repositories\Exams\ExamRepository;

class ExamService extends Services
{
    protected $examRepository;

    public function __construct(ExamRepository $examRepository)
    {
        $this->examRepository = $examRepository;
    }

    public function getExamsForSubscribedPackages($user, $limit)
    {

        // Get data from the repository
        $latestSubscriptionDate = $this->getLatestSubscriptionDate($user);
        $activePackageIds = $this->getActivePackageIds($user);

        // Return false if no active subscriptions or packages
        if (!$latestSubscriptionDate || $activePackageIds->isEmpty()) {
            return false;
        }

        $exams = $this->examRepository->getExamsBySubscribedPackages($limit, $activePackageIds, $user);

        $currentAttempts = $this->examRepository->getCurrentAttempts($user, $latestSubscriptionDate);

        return [
            'exams' => $exams,
            'current_attempts' => $currentAttempts,
        ];
    }

    public function getSubscribedPackageExamQuestions($user, $examId)
    {
        $exam = $this->examRepository->getExamWithCategoryAndQuestions($examId);

        if (!$exam) {
            $this->setError('Exam not found');
            return false;
        }

        $activePackageIds = $this->getActivePackageIds($user);

        if (!$this->examRepository->isExamInSubscribedPackages($user, $examId, $activePackageIds)) {
            $this->setError('You are not subscribed to this exam\'s package');
            return false;
        }

        $latestSubscriptionDate = $this->getLatestSubscriptionDate($user);

        $attemptsUsed = $this->examRepository->getAttemptsUsed($user, $examId, $latestSubscriptionDate);

        if ($attemptsUsed >= $exam->attempt_num) {
            $this->setError('You have exhausted your attempts');
            return false;
        }

        // Create a result record to log this attempt
        $resultRecord = $this->examRepository->createResultRecord($user, $exam, $attemptsUsed);

        $this->recountQuestionArrangements($exam);

        return [
            'exam' => $exam,
            'settings' => $this->prepareExamSettings($exam, $resultRecord->id),
        ];
    }

    public function getTestExam()
    {
        $exam = $this->examRepository->getExamWithCategoryAndQuestions($this->examRepository->getTestExam()->test_exam_id);

        if (!$exam) {
            $this->setError('Exam not found');
            return false;
        }

        $this->recountQuestionArrangements($exam);

        return [
            'exam' => $exam,
            'settings' => $this->prepareExamSettings($exam),
        ];
    }

    private function recountQuestionArrangements($exam)
    {
        $exam->examCategory->each(function ($category) {
            $category->questions->each(function ($question, $index) {
                $question->arrangment = $index;
            });
        });
    }

    private function prepareExamSettings($exam, $resultRecordId = null)
    {
        return [
            "id" => $exam->id,
            "name" => $exam->examName,
            "description" => $exam->description,
            "photo" => $exam->photo,
            'questions_num' => $exam->questions_num,
            'attempt_num' => $exam->attempt_num,
            'duration_in_minutes' => $exam->duration_in_minutes,
            'exam_category_auto_move' => $exam->exam_category_auto_move,
            'result_record' => $resultRecordId,
        ];
    }
    public function getLatestSubscriptionDate($user)
    {
        return $this->examRepository->getLatestSubscriptionDate($user);
    }
    public function getActivePackageIds($user)
    {
        return $this->examRepository->getActivePackageIds($user);
    }
    public function getExamsBySubscribedPackagesWithResults($user, $activePackageId, $latestSubscriptionDate)
    {
        return $this->examRepository->getExamsBySubscribedPackagesWithResults($user, $activePackageId, $latestSubscriptionDate);
    }

    public function updateResultRecord($request)
    {
        $rules = [
            'result_record' => ['required', 'integer', 'exists:results,id'],
            'passed_exam' => ['required', 'in:0,1'],
            'json_score' => ['nullable'],
            'total_right_questions' => ['required', 'integer', 'min:0'],
            'total_wrong_questions' => ['required', 'integer', 'min:0'],
            'total_skiped_questions' => ['required', 'integer', 'min:0'],
            'total_not_answered_questions' => ['required', 'integer', 'min:0'],
        ];

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());
            return false;
        }

        return $this->examRepository->updateResultRecord($request);
    }
}
