<?php

namespace App\Http\Controllers\Admin;

use Carbon\Carbon;
use App\Models\Offer;
use App\Models\Country;
use App\Models\Package;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\StoreOfferRequest;
use App\Http\Requests\UpdateOfferRequest;

class OfferController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index($packageId)
    {
        $countryCode = session('admin_country_code');

        if (!Package::where('id', $packageId)->exists()) {
            return;
        }

        $offers = Offer::when($countryCode, function ($query) use ($countryCode) {
            return $query->where('country_' . $countryCode, 1);
        })->where('package_id', $packageId)->latest()->get();

        return view('admin.Offer.index', compact('offers', 'packageId'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \App\Http\Requests\StoreOfferRequest  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreOfferRequest $request)
    {
        $validCountryCodes = Country::pluck('code')->toArray();

        $offer = new Offer;
        $offer->title_ar = $request->title_ar;
        $offer->title_en = $request->title_en;
        $offer->title_nl = $request->title_nl;
        $offer->package_id  = $request->package_id;
        $offer->start_date = $request->start_date;
        $offer->end_date = $request->end_date;
        $offer->discount_amount = $request->discount_amount;

        foreach ($validCountryCodes as $code) {
            $columnName = 'country_' . $code;
            $offer->$columnName = $request->has('country_' . $code) ? true : false;
        }

        $offer->save();
        return redirect()->back();
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Offer  $offer
     * @return \Illuminate\Http\Response
     */
    public function show(Offer $offer)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Offer  $offer
     * @return \Illuminate\Http\Response
     */
    public function edit(Offer $offer)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \App\Http\Requests\UpdateOfferRequest  $request
     * @param  \App\Models\Offer  $offer
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateOfferRequest $request, Offer $offer)
    {
        $selectedCountries = collect(Country::pluck('code'))
            ->mapWithKeys(function ($code) use ($request) {
                return ["country_$code" => $request->has("country_$code")];
            })
            ->filter()
            ->keys()
            ->toArray();

        $conflictingCountries = Offer::where('package_id', $offer->package_id)
            ->where('id', '!=', $offer->id)
            ->where(function ($query) {
                $query->whereNull('end_date')
                    ->orWhere('end_date', '>=', now());
            })
            ->get()
            ->flatMap(function ($otherOffer) use ($selectedCountries) {
                return collect($selectedCountries)->filter(function ($country) use ($otherOffer) {
                    return $otherOffer->{$country} == true;
                });
            })
            ->unique()
            ->values();

        if ($conflictingCountries->isNotEmpty()) {
            return redirect()->back()
                ->withErrors([
                    'country_conflict' => trans('messages.country_conflict_error'),
                    'conflicting_countries' => $conflictingCountries
                ]);
        }

        $offer->title_ar = $request->title_ar;
        $offer->title_en = $request->title_en;
        $offer->title_nl = $request->title_nl;
        $offer->package_id = $request->package_id;
        $offer->start_date = $request->start_date;
        $offer->end_date = $request->end_date;
        $offer->discount_amount = $request->discount_amount;

        foreach (Country::pluck('code') as $code) {
            $column = "country_$code";
            $offer->$column = $request->has($column);
        }

        $offer->save();
        return redirect()->back();
    }

    public function deactivateOffer(Offer $offer)
    {
        if ($offer->end_date && $offer->end_date <= Carbon::now()) {
            session()->flash('errorNotif', trans('messages.offer already deactivated'));
            return redirect()->route('package.index');
        }

        $offer->end_date = Carbon::yesterday();
        $offer->save();

        session()->flash('notif', trans('messages.offer deactivated successfully'));
        return redirect()->back();
    }

    public function createOffer(Request $request)
    {
        $package = Package::findOrFail($request->id);
        $countryCodes = Country::pluck('code')->toArray();

        $activeOffers = Offer::where('package_id', $package->id)
            ->where(function ($query) {
                $query->whereNull('end_date')
                    ->orWhere('end_date', '>=', now());
            })
            ->get();

        $disabledCountries = collect($countryCodes)
            ->filter(function ($code) use ($activeOffers) {
                $column = "country_{$code}";
                foreach ($activeOffers as $offer) {
                    if ($offer->{$column} != 0) {
                        return false;
                    }
                }
                return true;
            })
            ->values()
            ->toArray();

        $countriesToBeSelected = Country::whereIn('code', $disabledCountries)->get();

        return view('admin.Offer.create', [
            'package' => $package,
            'countries' => $countriesToBeSelected,
        ]);
    }

    public function updateOffer(Request $request, $package, $offer)
    {
        $package = Package::findOrFail($package);
        $offer = Offer::findOrFail($offer);

        return view('admin.Offer.edit', [
            'package' => $package,
            'offer' => $offer,
            'countries' => Country::all(),
        ]);
    }

    public function getOfferDiscount(Request $request)
    {
        return Offer::find($request->id)->discount_amount;
    }
}
